/*
 * Decompiled with CFR 0.152.
 */
package com.skynex.mylands.manager;

import com.skynex.mylands.config.ConfigManager;
import com.skynex.mylands.config.Messages;
import com.skynex.mylands.database.DatabaseManager;
import com.skynex.mylands.database.LandRepository;
import com.skynex.mylands.gui.ConfigurableLandMenuGui;
import com.skynex.mylands.gui.GuiManager;
import com.skynex.mylands.model.JobRequirement;
import com.skynex.mylands.model.Land;
import com.skynex.mylands.model.LandLevel;
import com.skynex.mylands.service.EconomyService;
import com.skynex.mylands.service.GradeService;
import com.skynex.mylands.service.JobsService;
import com.skynex.mylands.service.LandService;
import com.skynex.mylands.util.PluginLogger;
import com.skynex.mylands.util.ZoneVisualizer;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.OfflinePlayer;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;

public class LandManager {
    private final Plugin plugin;
    private final ConfigManager configManager;
    private final DatabaseManager databaseManager;
    private final LandRepository landRepository;
    private final LandService landService;
    private final EconomyService economyService;
    private final JobsService jobsService;
    private final GradeService gradeService;
    private final ZoneVisualizer zoneVisualizer;
    private GuiManager guiManager;
    private final Set<UUID> awaitingFarmName = ConcurrentHashMap.newKeySet();

    public LandManager(@NotNull Plugin plugin) {
        this.plugin = plugin;
        this.configManager = new ConfigManager(plugin);
        this.databaseManager = new DatabaseManager(plugin, this.configManager.getDatabaseConfig());
        this.landRepository = new LandRepository(this.databaseManager);
        this.landService = new LandService(this.landRepository, this.configManager.getInitialSize(), this.configManager.getHeightLimit(), this.configManager.getDepthLimit(), this.configManager.getMaxFarmSize(), this.configManager.getSizeIncreasePerLevel(), this.configManager.getMaxLevel(), this.configManager.getMaxCoOwners());
        this.economyService = new EconomyService(plugin, this.configManager.isEconomyEnabled());
        this.jobsService = new JobsService(this.configManager.isJobsEnabled(), this.configManager.getJobsProvider());
        this.gradeService = new GradeService(10);
        this.zoneVisualizer = new ZoneVisualizer(plugin);
        if (this.jobsService.isEnabled()) {
            this.jobsService.validateJobNames(this.configManager.getAllConfiguredJobNames());
        }
        PluginLogger.info("LandManager initialized", new Object[0]);
    }

    public CompletableFuture<Void> initialize() {
        return ((CompletableFuture)this.databaseManager.initialize(this.configManager.getDatabaseConfig()).thenCompose(v -> this.landService.initialize())).thenRun(() -> PluginLogger.info("LandManager fully initialized", new Object[0]));
    }

    public CompletableFuture<Void> shutdown() {
        return CompletableFuture.runAsync(() -> {
            this.zoneVisualizer.shutdown();
            this.databaseManager.shutdown();
            PluginLogger.info("LandManager shutdown complete", new Object[0]);
        });
    }

    public void reload() {
        this.configManager.reload();
        PluginLogger.info("Configuration reloaded", new Object[0]);
    }

    public CompletableFuture<Land> createLand(@NotNull Player player, @NotNull String landName) {
        UUID playerId = player.getUniqueId();
        if (!this.isValidLandName(landName)) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Invalid land name"));
        }
        if (this.economyService.isEnabled()) {
            return this.economyService.has((OfflinePlayer)player, this.configManager.getCreationCost()).thenCompose(has -> {
                if (!has.booleanValue()) {
                    return CompletableFuture.failedFuture(new IllegalStateException("Insufficient funds"));
                }
                return this.economyService.withdraw((OfflinePlayer)player, this.configManager.getCreationCost()).thenCompose(success -> {
                    if (!success.booleanValue()) {
                        return CompletableFuture.failedFuture(new IllegalStateException("Failed to withdraw funds"));
                    }
                    return this.landService.createLand(playerId, landName, player.getLocation()).thenApply(land -> {
                        this.gradeService.initializeDefaultGrades(playerId);
                        return land;
                    });
                });
            });
        }
        return this.landService.createLand(playerId, landName, player.getLocation()).thenApply(land -> {
            this.gradeService.initializeDefaultGrades(playerId);
            return land;
        });
    }

    public CompletableFuture<Boolean> deleteLand(@NotNull Player player) {
        return this.landService.getLand(player.getUniqueId()).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                return CompletableFuture.completedFuture(false);
            }
            Land land = (Land)landOpt.get();
            int refundAmount = this.configManager.calculateDeletionRefund(land.level());
            return this.landService.deleteLand(player.getUniqueId()).thenCompose(success -> {
                if (success.booleanValue() && this.economyService.isEnabled()) {
                    return this.economyService.deposit((OfflinePlayer)player, refundAmount).thenApply(v -> {
                        player.sendMessage(this.getMessages().get("delete_success", Map.of("refund", String.valueOf(refundAmount))));
                        return true;
                    });
                }
                return CompletableFuture.completedFuture(success);
            });
        });
    }

    public CompletableFuture<Optional<Land>> getLand(@NotNull UUID playerId) {
        return this.landService.getLand(playerId);
    }

    public CompletableFuture<Optional<Land>> getLandForPlayer(@NotNull UUID playerId) {
        return this.landService.getLand(playerId).thenCompose(landOpt -> {
            if (landOpt.isPresent()) {
                return CompletableFuture.completedFuture(landOpt);
            }
            return CompletableFuture.supplyAsync(() -> this.landService.getAllLands().stream().filter(land -> land.isCoOwner(playerId)).findFirst());
        });
    }

    public CompletableFuture<Land> upgradeLand(@NotNull Player player) {
        UUID playerId = player.getUniqueId();
        return ((CompletableFuture)this.landService.getLand(playerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                return CompletableFuture.failedFuture(new IllegalStateException("No land found"));
            }
            Land land = (Land)landOpt.get();
            int nextLevel = land.level() + 1;
            Optional<LandLevel> levelOpt = this.configManager.getLandLevel(nextLevel);
            if (levelOpt.isEmpty()) {
                return CompletableFuture.failedFuture(new IllegalStateException("Max level reached"));
            }
            LandLevel nextLevelConfig = levelOpt.get();
            if (this.jobsService.isEnabled() && !nextLevelConfig.jobRequirements().isEmpty()) {
                return this.checkJobRequirements(player, nextLevelConfig).thenCompose(meetsReqs -> {
                    if (!meetsReqs.booleanValue()) {
                        return CompletableFuture.failedFuture(new IllegalStateException("Job requirements not met"));
                    }
                    return this.processUpgrade(player, playerId, nextLevelConfig);
                });
            }
            return this.processUpgrade(player, playerId, nextLevelConfig);
        })).thenApply(upgradedLand -> {
            this.zoneVisualizer.displayWithParticles(player, upgradedLand.cuboid(), this.configManager.getZoneDisplayParticle(), 10);
            return upgradedLand;
        });
    }

    private CompletableFuture<Boolean> checkJobRequirements(@NotNull Player player, @NotNull LandLevel level) {
        ArrayList<CompletionStage> checks = new ArrayList<CompletionStage>();
        for (JobRequirement req : level.jobRequirements()) {
            CompletionStage check = this.jobsService.getJobLevel(player, req.jobName()).thenApply(currentLevel -> {
                boolean meets;
                boolean bl = meets = currentLevel >= req.requiredLevel();
                if (!meets) {
                    player.sendMessage(this.getMessages().jobsLevelTooLow((int)currentLevel, req.jobName(), level.level()));
                }
                return meets;
            });
            checks.add(check);
        }
        return CompletableFuture.allOf(checks.toArray(new CompletableFuture[0])).thenApply(v -> checks.stream().allMatch(CompletableFuture::join));
    }

    private CompletableFuture<Land> processUpgrade(@NotNull Player player, @NotNull UUID playerId, @NotNull LandLevel level) {
        if (this.economyService.isEnabled()) {
            return this.economyService.has((OfflinePlayer)player, level.upgradeCost()).thenCompose(has -> {
                if (!has.booleanValue()) {
                    return CompletableFuture.failedFuture(new IllegalStateException("Insufficient funds"));
                }
                return this.economyService.withdraw((OfflinePlayer)player, level.upgradeCost()).thenCompose(success -> {
                    if (!success.booleanValue()) {
                        return CompletableFuture.failedFuture(new IllegalStateException("Failed to withdraw funds"));
                    }
                    return this.landService.upgradeLand(playerId, level);
                });
            });
        }
        return this.landService.upgradeLand(playerId, level);
    }

    public CompletableFuture<Land> setHome(@NotNull Player player, @NotNull Location location) {
        return this.landService.setHome(player.getUniqueId(), location);
    }

    public CompletableFuture<Boolean> teleportHome(@NotNull Player player) {
        return this.landService.getLand(player.getUniqueId()).thenApply(landOpt -> {
            if (landOpt.isEmpty()) {
                return false;
            }
            Land land = (Land)landOpt.get();
            if (land.home() == null) {
                return false;
            }
            player.teleport(land.home());
            return true;
        });
    }

    public CompletableFuture<Land> setVisitsOpen(@NotNull UUID ownerId, boolean open) {
        return this.landService.setVisitsOpen(ownerId, open);
    }

    public CompletableFuture<Boolean> visitLand(@NotNull Player visitor, @NotNull UUID ownerId) {
        return this.landService.getLand(ownerId).thenApply(landOpt -> {
            if (landOpt.isEmpty()) {
                return false;
            }
            Land land = (Land)landOpt.get();
            if (!land.visitsOpen()) {
                return false;
            }
            if (land.home() == null) {
                return false;
            }
            visitor.teleport(land.home());
            return true;
        });
    }

    public void showZone(@NotNull Player player, @NotNull String displayType) {
        this.landService.getLand(player.getUniqueId()).thenAccept(landOpt -> {
            if (landOpt.isEmpty()) {
                return;
            }
            Land land = (Land)landOpt.get();
            if (displayType.equalsIgnoreCase("particles")) {
                this.zoneVisualizer.displayWithParticles(player, land.cuboid(), this.configManager.getZoneDisplayParticle(), this.configManager.getZoneDisplayDuration());
            } else if (displayType.equalsIgnoreCase("blocks")) {
                this.zoneVisualizer.displayWithBlocks(player, land.cuboid(), this.configManager.getZoneDisplayBlock(), this.configManager.getZoneDisplayDuration());
            }
        });
    }

    public boolean inviteCoOwner(@NotNull UUID ownerId, @NotNull UUID targetId) {
        return this.landService.inviteCoOwner(ownerId, targetId);
    }

    public CompletableFuture<Land> acceptInvitation(@NotNull UUID playerId) {
        return this.landService.acceptInvitation(playerId);
    }

    public CompletableFuture<Land> removeCoOwner(@NotNull UUID ownerId, @NotNull UUID coOwnerId) {
        return this.landService.removeCoOwner(ownerId, coOwnerId);
    }

    public CompletableFuture<Void> quitLand(@NotNull UUID playerId) {
        return this.landService.quitLand(playerId);
    }

    public boolean hasPendingInvitation(@NotNull UUID playerId) {
        return this.landService.hasPendingInvitation(playerId);
    }

    public boolean hasAccessToLocation(@NotNull UUID playerId, @NotNull Location location) {
        return this.landService.hasAccessToLocation(playerId, location);
    }

    public Land getLandAtLocation(@NotNull Location location) {
        return this.landService.getLandAtLocation(location);
    }

    public void addAwaitingLandName(@NotNull UUID playerId) {
        this.awaitingFarmName.add(playerId);
        Bukkit.getScheduler().runTaskLater(this.plugin, () -> this.awaitingFarmName.remove(playerId), 600L);
    }

    public boolean isAwaitingLandName(@NotNull UUID playerId) {
        return this.awaitingFarmName.contains(playerId);
    }

    public void removeAwaitingLandName(@NotNull UUID playerId) {
        this.awaitingFarmName.remove(playerId);
    }

    public boolean isValidLandName(@NotNull String name) {
        int maxLength = this.configManager.getMaxFarmNameLength();
        return name != null && name.matches("^[a-zA-Z0-9_\\- ]{1," + maxLength + "}$") && !name.isBlank();
    }

    public void setGuiManager(@NotNull GuiManager guiManager) {
        this.guiManager = guiManager;
    }

    public void openLandMenu(@NotNull Player player, @NotNull Land land) {
        if (this.guiManager == null) {
            player.sendMessage("\u00a7cGUI system not initialized!");
            return;
        }
        this.guiManager.openGui(player, new ConfigurableLandMenuGui(this.guiManager, this.configManager.getGuiConfig(), land, this.landService, this.gradeService));
    }

    public ConfigManager getConfigManager() {
        return this.configManager;
    }

    public Messages getMessages() {
        return this.configManager.getMessages();
    }

    public EconomyService getEconomyService() {
        return this.economyService;
    }

    public JobsService getJobsService() {
        return this.jobsService;
    }

    public GradeService getGradeService() {
        return this.gradeService;
    }

    public Plugin getPlugin() {
        return this.plugin;
    }

    public GuiManager getGuiManager() {
        return this.guiManager;
    }
}

